/**
 * \file helper.c
 *
 * \brief : SDC helper functions mainly used by sdc_ctl and tests
 * <b>Normal applications should not require these functions </b>
 *
 * \author Christoph Gellner (cgellner@de.adit-jv.com)
 *
 * \copyright (c) 2015 Advanced Driver Information Technology.
 * This code is developed by Advanced Driver Information Technology.
 * Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
 * All rights reserved.
 *
 *
 ***********************************************************************/

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <private/sdc_helper.h>
#include <private/sdc_intern.h>

#define SDC_HELPER_MECHANISM_NAME_MAX 30

struct sdc_operation_info_funcs {
    void *(*init)(void);
    bool (*next)(void *_iter);
    void (*free)(void *_iter);
    sdc_error_t (*set_type)(void *_iter, void *allocated_type);
    sdc_error_t (*get_name)(void *_iter, char *name, size_t name_len);
    sdc_error_t (*get_key_lens_fmt)(void *_iter,
                                    sdc_key_fmt_t *sup_key_fmt_protect, sdc_key_fmt_t *sup_key_fmt_unprotect,
                                    sdc_key_len_bmsk_t *sup_key_lens, sdc_key_len_t *dflt_key_len);
};

typedef struct {
    sdc_wrap_unwrap_alg_t alg;
    sdc_wrap_unwrap_blk_t blk;
    sdc_wrap_unwrap_type_t *type;
} sdc_helper_wrap_mechanism_t;

static void *sdc_helper_wrap_mechanism_init(void)
{
    sdc_error_t err;
    sdc_helper_wrap_mechanism_t *iter = malloc(sizeof(sdc_helper_wrap_mechanism_t));

    if (!iter)
        return NULL;

    iter->alg = SDC_WRAP_ALG_INVALID;
    iter->blk = SDC_WRAP_BLK_INVALID;

    err = sdc_wrap_unwrap_type_alloc(&iter->type);
    if (err != SDC_OK) {
        free(iter);
        return NULL;
    }

    return (void*)iter;
}

static bool sdc_helper_wrap_mechanism_next(void *_iter)
{
    sdc_helper_wrap_mechanism_t *iter = (sdc_helper_wrap_mechanism_t*)_iter;
    sdc_error_t err;
    bool valid;

    valid = false;

    while (valid == false) {

        if (iter->alg == SDC_WRAP_ALG_INVALID) {
            iter->alg = SDC_WRAP_ALG_FIRST;
        }

        if (iter->blk == SDC_WRAP_BLK_INVALID) {
            iter->blk = SDC_WRAP_BLK_FIRST;
        } else {
            iter->blk++;
            if (iter->blk >= SDC_WRAP_BLK_END) {
                iter->blk = SDC_WRAP_BLK_FIRST;

                iter->alg++;
            }
        }

        if (iter->alg >= SDC_WRAP_ALG_END) {
            break;
        }

        err = sdc_wrap_unwrap_type_set_algorithm(iter->type, iter->alg);
        if (err == SDC_OK) {
            err = sdc_wrap_unwrap_type_set_block_mode(iter->type, iter->blk);
        }
        if (err == SDC_OK) {
            valid = true;
        }
    }

    return valid;
}

static void sdc_helper_wrap_mechanism_free(void *_iter)
{
    sdc_helper_wrap_mechanism_t *iter = (sdc_helper_wrap_mechanism_t*)_iter;

    sdc_wrap_unwrap_type_free(iter->type);
    free (iter);
}

static sdc_error_t sdc_helper_wrap_mechanism_set_type(void *_iter, void *allocated_type)
{
    sdc_error_t err;
    sdc_helper_wrap_mechanism_t *iter = (sdc_helper_wrap_mechanism_t*)_iter;
    sdc_wrap_unwrap_type_t *type = (sdc_wrap_unwrap_type_t *)allocated_type;

    err = sdc_wrap_unwrap_type_set_algorithm(type, iter->alg);
    if (err == SDC_OK)
        err = sdc_wrap_unwrap_type_set_block_mode(type, iter->blk);

    return err;
}

static void sdc_helper_wrap_alg_blk_name(sdc_wrap_unwrap_alg_t alg, sdc_wrap_unwrap_blk_t blk, char *str, size_t str_len)
{
    snprintf(str, str_len, "%s_%s",
             sdc_wrap_unwrap_algorithm_name(alg),
             sdc_wrap_unwrap_block_mode_name(blk));
}

static sdc_error_t sdc_helper_wrap_mechanism_get_name(void *_iter, char *name, size_t name_len)
{
    sdc_helper_wrap_mechanism_t *iter = (sdc_helper_wrap_mechanism_t*)_iter;

    sdc_helper_wrap_alg_blk_name(iter->alg, iter->blk, name, name_len);

    return SDC_OK;
}

static sdc_error_t sdc_helper_wrap_mechanism_get_key_lens_fmt(void *_iter,
                                                              sdc_key_fmt_t *sup_key_fmt_protect, sdc_key_fmt_t *sup_key_fmt_unprotect,
                                                              sdc_key_len_bmsk_t *sup_key_lens, sdc_key_len_t *dflt_key_len)
{
    sdc_helper_wrap_mechanism_t *iter = (sdc_helper_wrap_mechanism_t*)_iter;
    sdc_error_t err;

    err = sdc_wrap_unwrap_get_key_key_lens_fmt(iter->type,
                                               sup_key_fmt_protect, sup_key_fmt_unprotect,
                                               sup_key_lens, dflt_key_len);

    return err;
}

static const sdc_operation_info_funcs_t sdc_wrap_info_funcs = {
    &sdc_helper_wrap_mechanism_init,
    &sdc_helper_wrap_mechanism_next,
    &sdc_helper_wrap_mechanism_free,
    &sdc_helper_wrap_mechanism_set_type,
    &sdc_helper_wrap_mechanism_get_name,
    &sdc_helper_wrap_mechanism_get_key_lens_fmt
};


typedef struct {
    sdc_encrypt_decrypt_alg_t alg;
    sdc_encrypt_decrypt_blk_t blk;
    sdc_encrypt_decrypt_type_t *type;
} sdc_helper_encrypt_mechanism_t;

static void *sdc_helper_encrypt_mechanism_init(void)
{
    sdc_error_t err;
    sdc_helper_encrypt_mechanism_t *iter = malloc(sizeof(sdc_helper_encrypt_mechanism_t));

    if (!iter)
        return NULL;

    iter->alg = SDC_ENCDEC_ALG_INVALID;
    iter->blk = SDC_ENCDEC_BLK_INVALID;

    err = sdc_encrypt_decrypt_type_alloc(&iter->type);
    if (err != SDC_OK) {
        free(iter);
        return NULL;
    }

    return (void*)iter;
}

static bool sdc_helper_encrypt_mechanism_next(void *_iter)
{
    sdc_helper_encrypt_mechanism_t *iter = (sdc_helper_encrypt_mechanism_t*)_iter;
    sdc_error_t err;
    bool valid;

    valid = false;

    while (valid == false) {

        if (iter->alg == SDC_ENCDEC_ALG_INVALID) {
            iter->alg = SDC_ENCDEC_ALG_FIRST;
        }

        if (iter->blk == SDC_ENCDEC_BLK_INVALID) {
            iter->blk = SDC_ENCDEC_BLK_FIRST;
        } else {
            iter->blk++;
            if (iter->blk >= SDC_ENCDEC_BLK_END) {
                iter->blk = SDC_ENCDEC_BLK_FIRST;

                iter->alg++;
            }
        }

        if (iter->alg >= SDC_ENCDEC_ALG_END) {
            break;
        }

        err = sdc_encrypt_decrypt_type_set_algorithm(iter->type, iter->alg);
        if (err == SDC_OK) {
            err = sdc_encrypt_decrypt_type_set_block_mode(iter->type, iter->blk);
        }
        if (err == SDC_OK) {
            valid = true;
        }
    }

    return valid;
}

static void sdc_helper_encrypt_mechanism_free(void *_iter)
{
    sdc_helper_encrypt_mechanism_t *iter = (sdc_helper_encrypt_mechanism_t*)_iter;

    sdc_encrypt_decrypt_type_free(iter->type);
    free (iter);
}

static sdc_error_t sdc_helper_encrypt_mechanism_set_type(void *_iter, void *allocated_type)
{
    sdc_error_t err;
    sdc_helper_encrypt_mechanism_t *iter = (sdc_helper_encrypt_mechanism_t*)_iter;
    sdc_encrypt_decrypt_type_t *type = (sdc_encrypt_decrypt_type_t *)allocated_type;

    err = sdc_encrypt_decrypt_type_set_algorithm(type, iter->alg);
    if (err == SDC_OK)
        err = sdc_encrypt_decrypt_type_set_block_mode(type, iter->blk);

    return err;
}

static void sdc_helper_encrypt_alg_blk_name(sdc_encrypt_decrypt_alg_t alg, sdc_encrypt_decrypt_blk_t blk, char *str, size_t str_len)
{
    const char* alg_name;
    const char* blk_name;

    alg_name = sdc_encrypt_decrypt_algorithm_name(alg);
    blk_name = sdc_encrypt_decrypt_block_mode_name(blk);

    /* RSA has no block mode */
    if (blk_name) {
        snprintf(str, str_len, "%s_%s",
                 alg_name,
                 blk_name);
    } else {
        snprintf(str, str_len, "%s",
                 alg_name);
    }
}


static sdc_error_t sdc_helper_encrypt_mechanism_get_name(void *_iter, char *name, size_t name_len)
{
    sdc_helper_encrypt_mechanism_t *iter = (sdc_helper_encrypt_mechanism_t*)_iter;

    sdc_helper_encrypt_alg_blk_name(iter->alg, iter->blk, name, name_len);

    return SDC_OK;
}

static sdc_error_t sdc_helper_encrypt_mechanism_get_key_lens_fmt(void *_iter,
                                                                 sdc_key_fmt_t *sup_key_fmt_protect, sdc_key_fmt_t *sup_key_fmt_unprotect,
                                                                 sdc_key_len_bmsk_t *sup_key_lens, sdc_key_len_t *dflt_key_len)
{
    sdc_helper_encrypt_mechanism_t *iter = (sdc_helper_encrypt_mechanism_t*)_iter;
    sdc_error_t err;

    err = sdc_encrypt_decrypt_get_key_key_lens_fmt(iter->type,
                                                   sup_key_fmt_protect, sup_key_fmt_unprotect,
                                                   sup_key_lens, dflt_key_len);

    return err;
}

static const sdc_operation_info_funcs_t sdc_encrypt_info_funcs = {
    &sdc_helper_encrypt_mechanism_init,
    &sdc_helper_encrypt_mechanism_next,
    &sdc_helper_encrypt_mechanism_free,
    &sdc_helper_encrypt_mechanism_set_type,
    &sdc_helper_encrypt_mechanism_get_name,
    &sdc_helper_encrypt_mechanism_get_key_lens_fmt
};

typedef struct {
    sdc_sign_verify_alg_t mac;
    sdc_sign_verify_hash_t hash;
    sdc_sign_verify_type_t *type;
} sdc_helper_sign_mechanism_t;

static void *sdc_helper_sign_mechanism_init(void)
{
    sdc_error_t err;
    sdc_helper_sign_mechanism_t *iter = malloc(sizeof(sdc_helper_sign_mechanism_t));

    if (!iter)
        return NULL;

    iter->mac = SDC_SIGNVER_ALG_INVALID;
    iter->hash = SDC_SIGNVER_HASH_INVALID;

    err = sdc_sign_verify_type_alloc(&iter->type);
    if (err != SDC_OK) {
        free(iter);
        return NULL;
    }

    return (void*)iter;
}

static bool sdc_helper_sign_mechanism_next(void *_iter)
{
    sdc_helper_sign_mechanism_t *iter = (sdc_helper_sign_mechanism_t*)_iter;
    sdc_error_t err;
    bool valid;

    valid = false;

    while (valid == false) {

        if (iter->mac == SDC_SIGNVER_ALG_INVALID) {
            iter->mac = SDC_SIGNVER_ALG_FIRST;
        }

        if (iter->hash == SDC_SIGNVER_HASH_INVALID) {
            iter->hash = SDC_SIGNVER_HASH_FIRST;
        } else {
            iter->hash++;
            if (iter->hash >= SDC_SIGNVER_HASH_END) {
                iter->hash = SDC_SIGNVER_HASH_FIRST;

                iter->mac++;
            }
        }

        if (iter->mac >= SDC_SIGNVER_ALG_END) {
            break;
        }

        err = sdc_sign_verify_type_set_alg(iter->type, iter->mac);
        if (err == SDC_OK) {
            err = sdc_sign_verify_type_set_hash(iter->type, iter->hash);
        }
        if (err == SDC_OK) {
            valid = true;
        }
    }

    return valid;
}

static void sdc_helper_sign_mechanism_free(void *_iter)
{
    sdc_helper_sign_mechanism_t *iter = (sdc_helper_sign_mechanism_t*)_iter;

    sdc_sign_verify_type_free(iter->type);
    free (iter);
}

static sdc_error_t sdc_helper_sign_mechanism_set_type(void *_iter, void *allocated_type)
{
    sdc_error_t err;
    sdc_helper_sign_mechanism_t *iter = (sdc_helper_sign_mechanism_t*)_iter;
    sdc_sign_verify_type_t *type = (sdc_sign_verify_type_t *)allocated_type;

    err = sdc_sign_verify_type_set_alg(type, iter->mac);
    if (err == SDC_OK)
        err = sdc_sign_verify_type_set_hash(type, iter->hash);

    return err;
}

static void sdc_helper_sign_mac_hash_name(sdc_sign_verify_alg_t mac, sdc_sign_verify_hash_t hash, char *str, size_t str_len)
{
    snprintf(str, str_len, "%s_%s",
             sdc_sign_verify_alg_name(mac),
             sdc_sign_verify_hash_name(hash));
}


static sdc_error_t sdc_helper_sign_mechanism_get_name(void *_iter, char *name, size_t name_len)
{
    sdc_helper_sign_mechanism_t *iter = (sdc_helper_sign_mechanism_t*)_iter;

    sdc_helper_sign_mac_hash_name(iter->mac, iter->hash, name, name_len);

    return SDC_OK;
}

static sdc_error_t sdc_helper_sign_mechanism_get_key_lens_fmt(void *_iter,
                                                              sdc_key_fmt_t *sup_key_fmt_protect, sdc_key_fmt_t *sup_key_fmt_unprotect,
                                                              sdc_key_len_bmsk_t *sup_key_lens, sdc_key_len_t *dflt_key_len)
{
    sdc_helper_sign_mechanism_t *iter = (sdc_helper_sign_mechanism_t*)_iter;
    sdc_error_t err;

    err = sdc_sign_verify_get_key_key_lens_fmt(iter->type,
                                               sup_key_fmt_protect, sup_key_fmt_unprotect,
                                               sup_key_lens, dflt_key_len);

    return err;
}

static const sdc_operation_info_funcs_t sdc_sign_info_funcs = {
    &sdc_helper_sign_mechanism_init,
    &sdc_helper_sign_mechanism_next,
    &sdc_helper_sign_mechanism_free,
    &sdc_helper_sign_mechanism_set_type,
    &sdc_helper_sign_mechanism_get_name,
    &sdc_helper_sign_mechanism_get_key_lens_fmt
};

typedef struct {
    sdc_dgst_hash_t hash;
    sdc_dgst_type_t *type;
} sdc_helper_dgst_mechanism_t;

static void *sdc_helper_dgst_mechanism_init(void)
{
    sdc_error_t err;
    sdc_helper_dgst_mechanism_t *iter = malloc(sizeof(sdc_helper_dgst_mechanism_t));

    if (!iter)
        return NULL;

    iter->hash = SDC_DGST_HASH_INVALID;

    err = sdc_dgst_type_alloc(&iter->type);
    if (err != SDC_OK) {
        free(iter);
        return NULL;
    }

    return (void*)iter;
}

static bool sdc_helper_dgst_mechanism_next(void *_iter)
{
    sdc_helper_dgst_mechanism_t *iter = (sdc_helper_dgst_mechanism_t*)_iter;
    sdc_error_t err;
    bool valid;

    valid = false;

    while (valid == false) {
        if (iter->hash == SDC_DGST_HASH_INVALID) {
            iter->hash = SDC_DGST_HASH_FIRST;
        } else {
            iter->hash++;
            if (iter->hash >= SDC_DGST_HASH_END) {
                iter->hash = SDC_DGST_HASH_FIRST;
                break;
            }
        }

        err = sdc_dgst_type_set_hash(iter->type, iter->hash);
        if (err == SDC_OK) {
            valid = true;
        }
    }

    return valid;
}

static void sdc_helper_dgst_mechanism_free(void *_iter)
{
    sdc_helper_dgst_mechanism_t *iter = (sdc_helper_dgst_mechanism_t*)_iter;

    sdc_dgst_type_free(iter->type);
    free (iter);
}

static sdc_error_t sdc_helper_dgst_mechanism_set_type(void *_iter, void *allocated_type)
{
    sdc_error_t err;
    sdc_helper_dgst_mechanism_t *iter = (sdc_helper_dgst_mechanism_t*)_iter;
    sdc_dgst_type_t *type = (sdc_dgst_type_t *)allocated_type;

    err = sdc_dgst_type_set_hash(type, iter->hash);

    return err;
}

static void sdc_helper_dgst_hash_name(sdc_dgst_hash_t hash, char *str, size_t str_len)
{
    snprintf(str, str_len, "%s",
             sdc_dgst_hash_name(hash));
}


static sdc_error_t sdc_helper_dgst_mechanism_get_name(void *_iter, char *name, size_t name_len)
{
    sdc_helper_dgst_mechanism_t *iter = (sdc_helper_dgst_mechanism_t*)_iter;

    sdc_helper_dgst_hash_name(iter->hash, name, name_len);

    return SDC_OK;
}

static const sdc_operation_info_funcs_t sdc_dgst_info_funcs = {
    &sdc_helper_dgst_mechanism_init,
    &sdc_helper_dgst_mechanism_next,
    &sdc_helper_dgst_mechanism_free,
    &sdc_helper_dgst_mechanism_set_type,
    &sdc_helper_dgst_mechanism_get_name,
    NULL
};

sdc_error_t sdc_helper_mechanism_iterator_init(sdc_mechanism_kinds_t kind, sdc_mechanism_iterator_t *iter)
{
    sdc_error_t err = SDC_OK;

    if (!iter) {
        return SDC_INVALID_PARAMETER;
    }

    iter->funcs = NULL;

    switch (kind) {
    case SDC_ENCRYPT_DECRYPT:
        iter->funcs = &sdc_encrypt_info_funcs;
        break;
    case SDC_WRAP_UNWRAP:
        iter->funcs = &sdc_wrap_info_funcs;
        break;
    case SDC_SIGN_VERIFY:
        iter->funcs = &sdc_sign_info_funcs;
        break;
    case SDC_DGST:
        iter->funcs = &sdc_dgst_info_funcs;
        break;
    default:
        err = SDC_INVALID_PARAMETER;
    }

    if (err == SDC_OK) {
        iter->type_specific_iterator = iter->funcs->init();
        if (!iter->type_specific_iterator)
            err = SDC_INTERNAL_ERROR;
    }

    if (err == SDC_OK) {
        err = sdc_helper_mechanism_iterator_next(iter);
    }

    return err;
}

static sdc_error_t sdc_helper_mechanism_check_iterator(sdc_mechanism_iterator_t *iter)
{
    if (!iter) {
        return SDC_INVALID_PARAMETER;
    }
    if (!iter->funcs) {
        return SDC_INVALID_PARAMETER;
    }
    if (!iter->type_specific_iterator) {
        return SDC_INVALID_PARAMETER;
    }

    return SDC_OK;
}

sdc_error_t sdc_helper_mechanism_iterator_next(sdc_mechanism_iterator_t *iter)
{
    sdc_error_t err;
    bool has_next;

    err = sdc_helper_mechanism_check_iterator(iter);

    if (err == SDC_OK) {
        has_next = iter->funcs->next(iter->type_specific_iterator);

        if (!has_next) {
            err = SDC_ALG_MODE_INVALID;
        }
    }

    return err;
}

void sdc_helper_mechanism_iterator_cleanup(sdc_mechanism_iterator_t *iter)
{
    sdc_error_t err;

    err = sdc_helper_mechanism_check_iterator(iter);
    if (err != SDC_OK)
        return;

    iter->funcs->free(iter->type_specific_iterator);
    iter->type_specific_iterator = NULL;
    iter->funcs = NULL;
}



sdc_error_t sdc_helper_mechanism_set_type(sdc_mechanism_iterator_t *iter, void *allocated_type)
{
    sdc_error_t err;

    err = sdc_helper_mechanism_check_iterator(iter);

    if (err == SDC_OK) {
        if (!allocated_type)
            err = SDC_INVALID_PARAMETER;
    }

    if (err == SDC_OK) {
        err = iter->funcs->set_type(iter->type_specific_iterator, allocated_type);
    }

    return err;
}


sdc_error_t sdc_helper_mechanism_get_name(sdc_mechanism_iterator_t *iter, char* name, size_t name_len)
{
    sdc_error_t err;

    err = sdc_helper_mechanism_check_iterator(iter);

    if (err == SDC_OK) {
        if (!name)
            err = SDC_INVALID_PARAMETER;
    }

    if (err == SDC_OK) {
        err = iter->funcs->get_name(iter->type_specific_iterator, name, name_len);
    }

    return err;
}

sdc_error_t sdc_helper_mechanism_get_key_info(sdc_mechanism_iterator_t *iter,
                                              sdc_key_fmt_t *sup_key_fmt_protect, sdc_key_fmt_t *sup_key_fmt_unprotect,
                                              sdc_key_len_bmsk_t *sup_key_lens, sdc_key_len_t *dflt_key_len)
{
    sdc_error_t err;

    err = sdc_helper_mechanism_check_iterator(iter);

    if (err == SDC_OK) {
        if (iter->funcs->get_key_lens_fmt != NULL)
            err = iter->funcs->get_key_lens_fmt(iter->type_specific_iterator,
                                                sup_key_fmt_protect, sup_key_fmt_unprotect,
                                                sup_key_lens, dflt_key_len);
        else
            err = SDC_KEYLEN_INVALID;
    }

    return err;
}

sdc_error_t sdc_helper_set_mechanism_by_name(sdc_mechanism_kinds_t kind, char *search_name, void *allocated_type)
{
    sdc_mechanism_iterator_t iter;
    char iter_mechanism[SDC_HELPER_MECHANISM_NAME_MAX];
    sdc_error_t err;

    if (!search_name)
        return SDC_INVALID_PARAMETER;

    err = sdc_helper_mechanism_iterator_init(kind, &iter);

    while (err == SDC_OK) {
        err = sdc_helper_mechanism_get_name(&iter, iter_mechanism, SDC_HELPER_MECHANISM_NAME_MAX);

        if(0 == strcmp(search_name, iter_mechanism)) {
            err = sdc_helper_mechanism_set_type(&iter, allocated_type);
            break;
        }

        if (err == SDC_OK) {
            err = sdc_helper_mechanism_iterator_next(&iter);
        }
    }

    sdc_helper_mechanism_iterator_cleanup(&iter);

    return err;
}

sdc_error_t sdc_helper_key_len_str(sdc_key_len_bmsk_t sup_key_lens, sdc_key_len_t dflt_key_len, char *keylen_info, size_t keylen_info_len)
{
    sdc_key_len_t keylen_val;
    size_t keylen_bits;
    sdc_error_t err;
    char *pos;
    size_t len;
    bool first = true;
    bool def;

    if (!keylen_info)
        return SDC_INVALID_PARAMETER;

    memset(keylen_info, 0, keylen_info_len);

    err = sdc_key_len_bmsk_first(sup_key_lens, &keylen_val);

    while ((err == SDC_OK) && (keylen_val != SDC_KEY_LEN_UNKNOWN)) {

        err = sdc_key_len_get_bits(keylen_val, &keylen_bits);

        def = (keylen_val == dflt_key_len);

        len = strlen(keylen_info);
        pos = keylen_info + len;
        snprintf(pos, (keylen_info_len-len),
                 first ? "%zu%s" : ",%zu%s",  keylen_bits, (def ? "(default)" : "") );
        first = false;

        err = sdc_key_len_bmsk_next(sup_key_lens, &keylen_val);
    }

    return err;
}

sdc_error_t sdc_helper_encrypt_decrypt_type_name(const sdc_encrypt_decrypt_type_t *type, char* typestr, size_t typestr_len)
{
    sdc_error_t err;
    sdc_encrypt_decrypt_alg_t alg = SDC_ENCDEC_ALG_INVALID;
    sdc_encrypt_decrypt_blk_t blk = SDC_ENCDEC_BLK_INVALID;

    err = sdc_encrypt_decrypt_type_get_algorithm(type, &alg);

    if (err == SDC_OK) {
        err = sdc_encrypt_decrypt_type_get_block_mode(type, &blk);
    }

    if (err == SDC_OK) {
        if (!typestr)
            err = SDC_INVALID_PARAMETER;
    }

    if (err == SDC_OK) {
        sdc_helper_encrypt_alg_blk_name(alg, blk, typestr, typestr_len);
    }

    return err;
}

sdc_error_t sdc_helper_wrap_unwrap_type_name(const sdc_wrap_unwrap_type_t *type, char* typestr, size_t typestr_len)
{
    sdc_error_t err;
    sdc_wrap_unwrap_alg_t alg = SDC_WRAP_ALG_INVALID;
    sdc_wrap_unwrap_blk_t blk = SDC_WRAP_BLK_INVALID;

    err = sdc_wrap_unwrap_type_get_algorithm(type, &alg);

    if (err == SDC_OK) {
        err = sdc_wrap_unwrap_type_get_block_mode(type, &blk);
    }

    if (err == SDC_OK) {
        if (!typestr)
            err = SDC_INVALID_PARAMETER;
    }

    if (err == SDC_OK) {
        sdc_helper_wrap_alg_blk_name(alg, blk, typestr, typestr_len);
    }

    return err;
}

sdc_error_t sdc_helper_sign_verify_type_name(const sdc_sign_verify_type_t *type, char* typestr, size_t typestr_len)
{
    sdc_error_t err;
    sdc_sign_verify_alg_t mac = SDC_SIGNVER_ALG_INVALID;
    sdc_sign_verify_hash_t hash = SDC_SIGNVER_HASH_INVALID;

    err = sdc_sign_verify_type_get_alg(type, &mac);

    if (err == SDC_OK) {
        err = sdc_sign_verify_type_get_hash(type, &hash);
    }

    if (err == SDC_OK) {
        if (!typestr)
            err = SDC_INVALID_PARAMETER;
    }

    if (err == SDC_OK) {
        sdc_helper_sign_mac_hash_name(mac, hash, typestr, typestr_len);
    }

    return err;
}
